<?php
/**
 *	WHMCS Payment Gateway - Coinify
 *
 *	@package     WHMCS
 *	@version     1.0.0
 *	@copyright   Copyright (c) ALC Media 2026
 *	@author      ALC Media <info@alc.media>
 *  @link        https://coinify.readme.io/docs/whmcs
 */

if (!defined('WHMCS')) {
    exit(header('Location: https://www.alc.media'));
}

use WHMCS\Billing\Invoice;
use WHMCS\Database\Capsule;
use WHMCS\Module\GatewaySetting;
use WHMCS\Billing\Payment\Transaction\History;
use WHMCS\Exception\Module\InvalidConfiguration;

function coinify_API(array $params) {
    $url = strpos($params['apiKey'], 'production_') !== false ? 'https://api.payment.coinify.com/v1/' : 'https://api.payment.sandbox.coinify.com/v1/';
    $json = [];
    $method = '';

    switch ($params['action']) {
        case 'Subaccount - Read Sub-account':
            $url .= 'subaccounts';
            $method = 'GET';
            break;
            
        case 'Payment Intent - Create Payment Intent':
            $url .= 'payment-intents';
            $method = 'POST';

            $invoiceID = $params['invoiceid'];

            if ($params['useInvoiceNumber']) {
                $invoice = Invoice::find($invoiceID);

                if (!$invoice) {
                    throw new Exception('Invoice Not Found');
                }

                $invoiceID = $invoice->invoicenum;
            }

            $json = [
                'amount' => (float) $params['amount'],
                'currency' => $params['currency'],
                'orderId' => (string) $invoiceID,
                'customerId' => (string) $params['clientdetails']['owner_user_id'],
                'customerEmail' => (string) $params['clientdetails']['email'],
                'pluginIdentifier' => 'WHMCS Coinify API Integration',
                'successUrl' => $params['returnurl'] . '&paymentsuccess=true',
                'failureUrl' => $params['returnurl'] . '&paymentfailed=true'
            ];
            break;

        case 'Payment Intent - Read Payment Intent':
            $url .= 'payment-intents/' . $params['paymentIntent'];
            $method = 'GET';
            break;

        default:
            throw new Exception('Invalid action: ' . $params['action']);
            break;
    }

    $curl = curl_init();
    curl_setopt($curl, CURLOPT_URL, $url);
    curl_setopt($curl, CURLOPT_CUSTOMREQUEST, $method);
    curl_setopt($curl, CURLOPT_TIMEOUT, 10);
    curl_setopt($curl, CURLOPT_FOLLOWLOCATION, 1);
    curl_setopt($curl, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($curl, CURLOPT_POSTREDIR, CURL_REDIR_POST_301);
    curl_setopt($curl, CURLOPT_SSLVERSION, CURL_SSLVERSION_TLSv1_2);
    curl_setopt($curl, CURLOPT_USERAGENT, 'WHMCS Coinify API Integration');
    curl_setopt($curl, CURLOPT_HTTPHEADER, [
        'Accept: application/json',
        'Content-Type: application/json',
        'X-API-KEY: ' . $params['apiKey']
    ]);

    if ($method === 'POST' || $method === 'PATCH') {
        curl_setopt($curl, CURLOPT_POSTFIELDS, json_encode($json));
    }

    $response = curl_exec($curl);
    $statusCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);

    if ($statusCode === 0) throw new Exception('cURL Error: ' . curl_error($curl));

    curl_close($curl);

    logModuleCall('Coinify', $url, !empty($json) ? json_encode($json) : '', $response);

    $response = json_decode($response, true);

    if (isset($response['errorMessage'])) {
        throw new Exception($response['errorMessage']);
    }

    return $response;
}

function coinify_MetaData() {
    return [
        'DisplayName' => 'Coinify',
        'APIVersion' => '1.1'
    ];
}

function coinify_config() {
    global $CONFIG;

    $webhookURL = rtrim($CONFIG['SystemURL'], '/') . '/modules/gateways/callback/coinify.php';

    $config = [
        'FriendlyName' => [
            'Type' => 'System',
            'Value' => 'Coinify'
        ],
        'apiKey' => [
            'FriendlyName' => 'API Key',
            'Type' => 'password',
            'Size' => '30',
            'Description' => 'Enter your API Key here. More information: <a href="https://coinify.readme.io/docs/authentication" target="_blank">here</a>.',
        ],
        'WebhookURL' => [
            'FriendlyName' => 'Webhook URL',
            'Description' => 'The webhook URL to be set on your Coinify account is: <a href="' . $webhookURL . '">' . $webhookURL . '</a>.<br>More information about webhooks can be found <a href="https://coinify.readme.io/docs/webhooks" target="_blank">here</a>.',
        ],
        'sharedSecret' => [
            'FriendlyName' => 'Shared Secret',
            'Type' => 'text',
            'Size' => '30',
            'Default' => sprintf(
                '%04x%04x-%04x-%04x-%04x-%04x%04x%04x',
                mt_rand(0, 0xffff),
                mt_rand(0, 0xffff),
                mt_rand(0, 0xffff),
                mt_rand(0, 0x0fff) | 0x4000,
                mt_rand(0, 0x3fff) | 0x8000,
                mt_rand(0, 0xffff),
                mt_rand(0, 0xffff),
                mt_rand(0, 0xffff)
            ),
            'Description' => 'Self generated UUID v4. This shared secret must match the webhook shared secret on your Coinify account.<br>You can use the default one or create a new one on your Dashboard. See more info <a href="https://coinify.readme.io/docs/webhooks" target="_blank">here</a>.'
        ],
        'useInvoiceNumber' => [
            'FriendlyName' => 'Use Invoice Number',
            'Type' => 'yesno',
            'Default' => false,
            'Description' => 'Tick to use the Invoice Number instead of the Invoice ID for the Coinify order reference.'
        ]
    ];

    return $config;
}

function coinify_config_validate(array $params = []) {
    if ($params['apiKey']) {
        $params['action'] = 'Subaccount - Read Sub-account';
        coinify_API($params);
    } else {
        throw new InvalidConfiguration('The API Key is invalid.');
    }
}

function coinify_link($params) {
    try {
        global $smarty, $_LANG;

        $transactions = History::where('gateway', 'coinify')
            ->where('invoice_id', $params['invoiceid'])
            ->orderBy('id', 'desc')
            ->get();

        foreach ($transactions as $transaction) {
            $params['action'] = 'Payment Intent - Read Payment Intent';
            $params['paymentIntent'] = $transaction->transaction_id;
            $checkout = coinify_API($params);

            $transaction->remoteStatus = $checkout['state'];
            $transaction->description = $checkout['stateReason'];
            $transaction->completed = ($checkout['state'] === 'completed');
            $transaction->save();
            
            if ($checkout['state'] === 'completed') {
                addInvoicePayment($params['invoiceid'], $checkout['id'], $checkout['amount'], 0, 'coinify');

                logTransaction('Coinify', $checkout, 'Invoice Paid', [
                    'history_id' => $transaction->id,
                ]);

                return '<script type="text/javascript">window.location.reload();</script>';
            }

            if ($checkout['state'] === 'pending' && $checkout['stateReason'] === 'pending_confirmations' && count($checkout['payments']) > 0) {
                return '<div class="alert alert-warning">' . $_LANG['invoicePaymentPendingCleared'] . '</div>';
            }
        }

        $params['action'] = 'Payment Intent - Create Payment Intent';
        $checkout = coinify_API($params);

        History::create([
            'gateway' => 'coinify',
            'invoice_id' => $params['invoiceid'],
            'transaction_id' => $checkout['id'],
            'remote_status' => 'pending',
            'description' => 'Payment Intent Created',
            'completed' => false
        ]);

        if (!$smarty) {
            $smarty = new WHMCS\Smarty();
        }

        $assetHelper = \DI::make('asset');

        $smarty->assign('WEB_ROOT', $assetHelper->getWebRoot());
        $smarty->assign('checkout', $checkout['paymentWindowUrl']);

        return $smarty->fetch(ROOTDIR . '/modules/gateways/coinify/btnPayNow.tpl');
    } catch (Exception $e) {
        return '<div class="alert alert-danger">' . $e->getMessage() . '</div>';
    }
}